﻿using System;
using System.Collections;
using System.Collections.Generic;

namespace Jeneva.Validation.Impl
{
    /// <summary>
    /// Represents basic set of validation routines
    /// </summary>
    public class JenevaValidationContext : IJenevaValidationContext
    {
        private readonly IMath math;
        private readonly IPathHelper pathHelper;
        private readonly IChecker checker;
        private readonly LinkedList<PathNode> path;
        private bool isValid;
        private bool isTargetValid;
        private bool isFieldValid;

        /// <summary>
        /// Name of the current validated field
        /// </summary>
        protected string fieldName;

        /// <summary>
        /// Value of the current validated field
        /// </summary>
        protected object fieldValue;

        /// <summary>
        /// List of registered failures
        /// </summary>
        protected IFailureList failures;

        /// <summary>
        /// Initializes new instance of the ValidationContext class
        /// </summary>
        /// <param name="context">Jeneva context</param>
        public JenevaValidationContext(IJenevaContext context)
        {
            this.math = context.Math;
            this.pathHelper = context.PathHelper;
            this.checker = context.Checker;
            this.path = this.pathHelper.CreateNew();
            this.isValid = true;
        }

        /// <summary>
        /// False, if at least on failure is recorded, otherwise true
        /// </summary>
        public bool IsValid
        {
            get { return this.isValid; }
        }

        /// <summary>
        /// False, if at least on failure is recorded for the curr, otherwise true
        /// </summary>
        public bool IsTargetValid
        {
            get { return this.isTargetValid; }
        }

        /// <summary>
        /// False, if at least on failure is recorded for the current field, otherwise true
        /// </summary>
        public bool IsFieldValid
        {
            get { return this.isFieldValid; }
        }

        /// <summary>
        /// Sets current target object
        /// </summary>
        /// <param name="value">target object</param>
        public void SetTarget(Dtobase value)
        {
            this.pathHelper.SetTopNodeTarget(this.path, value);
            this.isTargetValid = true;
        }

        /// <summary>
        /// Sets current index (used for indexed properties).
        /// For ex. current target is 'Children' and index is '7' and field 'Name' - failure would have this path - 'children[7].name'.
        /// </summary>
        /// <param name="index">index</param>
        public void SetIndex(int index)
        {
            this.path.Last.Value.Index = index;
        }

        /// <summary>
        /// Sets current field (property) name
        /// </summary>
        /// <param name="name">field property name</param>
        /// <param name="value">field value</param>
        public void SetField(string name, object value)
        {
            this.fieldValue = value;
            this.fieldName = name;
            this.isFieldValid = true;

            if (null != value)
            {
                Dtobase target = this.pathHelper.GetTopNodeTarget(this.path);
                target.AddAssignedField(this.fieldName);
            }
        }

        /// <summary>
        /// Propagates current field to path.
        /// For ex. if current field is 'User', call AddNested(), then call SetField("Name", null) - the failure path would be - User.Name.
        /// Usually after AddNested() call, SetTarget() must be called as well.
        /// </summary>
        public void AddNested()
        {
            this.path.AddLast(new PathNode(this.fieldName));
        }

        /// <summary>
        /// Goes one node back in path.
        /// </summary>
        public void RemoveNested()
        {
            this.path.RemoveLast();
        }

        /// <summary>
        /// Register failure with static key and message (ignoring the property path)
        /// </summary>
        /// <param name="key">failure key</param>
        /// <param name="msg">failure message</param>
        public void FailRoot(string key, string msg)
        {
            this.Fail(new Failure(key, msg));
        }

        /// <summary>
        /// Registers failure using current path
        /// </summary>
        /// <param name="msg">failure message</param>
        public void Fail(string msg)
        {
            this.Fail(new Failure(this.pathHelper.BuildPath(this.path, this.fieldName), msg));
        }

        /// <summary>
        /// Registers failure
        /// </summary>
        /// <param name="failure">failure object</param>
        public void Fail(Failure failure)
        {
            this.isFieldValid = false;
            this.isTargetValid = false;
            this.isValid = false;
            if (null == this.failures)
            {
                this.failures = new FailureList();
            }

            this.failures.Fail(failure);
        }

        /// <summary>
        /// True if current field is assigned, otherwise false
        /// </summary>
        /// <returns>true is assigned</returns>
        public bool IsAssigned()
        {
            Dtobase target = this.pathHelper.GetTopNodeTarget(this.path);
            return this.checker.IsAssigned(this.fieldName, target);
        }

        /// <summary>
        /// True is current field is null, otherwise false
        /// </summary>
        /// <returns>true is null</returns>
        public bool IsNull()
        {
            return this.checker.IsNull(this.fieldValue);
        }

        /// <summary>
        /// True if current field is correctly parsed, otherwise false
        /// </summary>
        /// <returns>true if valid</returns>
        public bool IsValidFormat()
        {
            Dtobase target = this.pathHelper.GetTopNodeTarget(this.path);
            return this.checker.IsValidFormat(this.fieldName, target);
        }

        /// <summary>
        /// Validates if current field is assigned (was present in incoming JSON)
        /// </summary>
        /// <param name="msg">failure message</param>
        public void Assigned(string msg)
        {
            if (!this.IsAssigned())
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value is not assigned by JSON deserializer
        /// </summary>
        /// <param name="msg">failure message</param>
        public void NotAssigned(string msg)
        {
            if (this.IsAssigned())
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value is correctly parsed
        /// </summary>
        /// <param name="msg">failure message</param>
        public void ValidFormat(string msg)
        {
            if (!this.IsValidFormat())
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value is null
        /// </summary>
        /// <param name="msg">failure message</param>
        public void Null(string msg)
        {
            if (!this.checker.IsNull(this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value is not null
        /// </summary>
        /// <param name="msg">failure message</param>
        public void NotNull(string msg)
        {
            if (this.checker.IsNull(this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value is equal to the value
        /// </summary>
        /// <param name="value">the value</param>
        /// <param name="msg">failure message</param>
        public void EqualTo<T>(T value, string msg) where T : IEquatable<T>
        {
            if (!this.checker.IsEqualTo<T>(value, (T)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value is not equal to the value
        /// </summary>
        /// <param name="value">the value</param>
        /// <param name="msg">failure message</param>
        public void NotEqualTo<T>(T value, string msg) where T : IEquatable<T>
        {
            if (this.checker.IsEqualTo<T>(value, (T)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value is equal to one of the values
        /// </summary>
        /// <param name="values">the values</param>
        /// <param name="msg">failure message</param>
        public void EqualToOneOf<T>(T[] values, string msg) where T : IEquatable<T>
        {
            if (!this.checker.IsEqualToOneOf<T>(values, (T)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value is not equal to any of the values
        /// </summary>
        /// <param name="values">the values</param>
        /// <param name="msg">failure message</param>
        public void NotEqualToAnyOf<T>(T[] values, string msg) where T : IEquatable<T>
        {
            if (this.checker.IsEqualToOneOf<T>(values, (T)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value is not string.Empty
        /// </summary>
        /// <param name="msg">failure message</param>
        public void StringNotEmpty(string msg)
        {
            if (null == this.fieldValue) return;
            if (this.checker.IsEmptyString((string)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value length (string.Length) is between min and max
        /// </summary>
        /// <param name="min">min string.Length</param>
        /// <param name="max">max string.Length</param>
        /// <param name="msg">failure message</param>
        public void StringLengthBetween(int min, int max, string msg)
        {
            if (null == this.fieldValue) return;
            if (!this.checker.IsLengthBetween(min, max, (string)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value (IComparable) is less or equal to m
        /// </summary>
        /// <param name="m">m</param>
        /// <param name="msg">failure message</param>
        public void LessOrEqualTo<T>(T m, string msg) where T : struct, IComparable<T>
        {
            if (null == this.fieldValue) return;
            if (!this.checker.IsLessOrEqualTo<T>(m, (T)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value (IComparable) is less than m
        /// </summary>
        /// <param name="m">m</param>
        /// <param name="msg">failure message</param>
        public void LessThan<T>(T m, string msg) where T : struct, IComparable<T>
        {
            if (null == this.fieldValue) return;
            if (this.checker.IsLessThan<T>(m, (T)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value (IComparable) is greater or equal to m
        /// </summary>
        /// <param name="m">m</param>
        /// <param name="msg">failure message</param>
        public void GreaterOrEqualTo<T>(T m, string msg) where T : struct, IComparable<T>
        {
            if (null == this.fieldValue) return;
            if (!this.checker.IsGreaterOrEqualTo<T>(m, (T)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value (IComparable) is greater than m
        /// </summary>
        /// <param name="m">m</param>
        /// <param name="msg">failure message</param>
        public void GreaterThan<T>(T m, string msg) where T : struct, IComparable<T>
        {
            if (null == this.fieldValue) return;
            if (!this.checker.IsGreaterThan<T>(m, (T)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value count (ICollection.Count) is between min and max count
        /// </summary>
        /// <param name="min">min count</param>
        /// <param name="max">max count</param>
        /// <param name="msg">failure message</param>
        public void CountIsBetween(int min, int max, string msg)
        {
            if (null == this.fieldValue) return;
            if (!this.checker.IsCountBetween(min, max, (ICollection)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Validates if current field value follows the regular expression
        /// </summary>
        /// <param name="expr">the regular expression</param>
        /// <param name="msg">failure message</param>
        public void Regex(string expr, string msg)
        {
            if (null == this.fieldValue) return;
            if (!this.checker.IsRegex(expr, (string)this.fieldValue))
            {
                this.Fail(msg);
            }
        }

        /// <summary>
        /// Throws ValidationException if at least one failure was registered
        /// </summary>
        public void Assert()
        {
            if (null != this.failures)
            {
                throw new ValidationException(this.failures);
            }
        }

        /// <summary>
        /// Gets Math helper object
        /// </summary>
        public IMath Math
        {
            get { return this.math; }
        }
    }
}